<?php
/* --------------------------------------------------------------
   WithdrawalRepository.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\Repository;

use Doctrine\DBAL\ConnectionException;
use Gambio\Admin\Withdrawal\Events\AllWithdrawalsRequested;
use Gambio\Admin\Withdrawal\Events\DeletedMultipleWithdrawals;
use Gambio\Admin\Withdrawal\Events\DeletedSingleWithdrawal;
use Gambio\Admin\Withdrawal\Events\DeletionOfMultipleWithdrawalsRequested;
use Gambio\Admin\Withdrawal\Events\DeletionOfSingleWithdrawalRequested;
use Gambio\Admin\Withdrawal\Events\FetchedAllWithdrawals;
use Gambio\Admin\Withdrawal\Events\FetchedSpecificWithdrawal;
use Gambio\Admin\Withdrawal\Events\SpecificWithdrawalRequested;
use Gambio\Admin\Withdrawal\Events\StorageOfMultipleWithdrawalsRequested;
use Gambio\Admin\Withdrawal\Events\StorageOfSingleWithdrawalRequested;
use Gambio\Admin\Withdrawal\Events\StoredMultipleWithdrawals;
use Gambio\Admin\Withdrawal\Events\StoredSingleWithdrawal;
use Gambio\Admin\Withdrawal\Exceptions\WithdrawalNotFoundException;
use Gambio\Admin\Withdrawal\Interfaces\Withdrawal;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalId;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalIds;
use Gambio\Admin\Withdrawal\Interfaces\Withdrawals;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalSqlCriteria;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalSqlPagination;
use Gambio\Admin\Withdrawal\Models\WithdrawalId as WithdrawalIdModel;
use Gambio\Admin\Withdrawal\Models\Withdrawals as WithdrawalsModel;
use InvalidArgumentException;
use Psr\EventDispatcher\EventDispatcherInterface;

/**
 * Class WithdrawalRepository
 *
 * @package Gambio\Admin\Withdrawal\Repository
 */
class WithdrawalRepository
{
    /**
     * @var WithdrawalMapper
     */
    private $mapper;
    
    /**
     * @var WithdrawalReader
     */
    private $reader;
    
    /**
     * @var WithdrawalWriter
     */
    private $writer;
    
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;
    
    
    /**
     * WithdrawalRepository constructor.
     *
     * @param WithdrawalMapper         $mapper
     * @param WithdrawalReader         $reader
     * @param WithdrawalWriter         $writer
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        WithdrawalMapper $mapper,
        WithdrawalReader $reader,
        WithdrawalWriter $writer,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->mapper          = $mapper;
        $this->reader          = $reader;
        $this->writer          = $writer;
        $this->eventDispatcher = $eventDispatcher;
    }
    
    
    /**
     * Returns a list of all withdrawals.
     *
     * @param WithdrawalSqlCriteria   $criteria
     * @param WithdrawalSqlPagination $pagination
     *
     * @return Withdrawals
     */
    public function getAll(
        WithdrawalSqlCriteria $criteria,
        WithdrawalSqlPagination $pagination
    ): Withdrawals {
        $event = AllWithdrawalsRequested::create($criteria, $pagination);
        $this->eventDispatcher->dispatch($event);
        
        $withdrawals     = [];
        $withdrawalsData = $this->reader->getAll($criteria, $pagination);
        foreach ($withdrawalsData as $withdrawalData) {
            $withdrawals[] = $this->mapper->mapWithdrawal($withdrawalData);
        }
        
        $event = FetchedAllWithdrawals::create(WithdrawalsModel::create(...$withdrawals));
        $this->eventDispatcher->dispatch($event);
        
        return $event->withdrawals();
    }
    
    
    /**
     * Returns the number of all withdrawals.
     *
     * @param WithdrawalSqlCriteria $criteria
     *
     * @return int
     */
    public function getTotalCount(WithdrawalSqlCriteria $criteria): int
    {
        return $this->reader->getTotalCount($criteria);
    }
    
    
    /**
     * Returns the withdrawal with the provided ID.
     *
     * @param WithdrawalId $id
     *
     * @return Withdrawal
     *
     * @throws WithdrawalNotFoundException
     */
    public function getById(WithdrawalId $id): Withdrawal
    {
        $event = SpecificWithdrawalRequested::create($id);
        $this->eventDispatcher->dispatch($event);
        
        $withdrawalData = $this->reader->getById($id);
        
        $event = FetchedSpecificWithdrawal::create($this->mapper->mapWithdrawal($withdrawalData));
        $this->eventDispatcher->dispatch($event);
        
        return $event->withdrawal();
    }
    
    
    /**
     * Stores the provided withdrawal.
     *
     * @param Withdrawal $withdrawal
     *
     * @return WithdrawalId
     */
    public function store(Withdrawal $withdrawal): WithdrawalId
    {
        $event = StorageOfSingleWithdrawalRequested::create($withdrawal);
        $this->eventDispatcher->dispatch($event);
        
        if ($withdrawal->id() !== null) {
            $this->writer->update($withdrawal);
            
            $event = StoredSingleWithdrawal::create(WithdrawalIdModel::create($withdrawal->id()));
            $this->eventDispatcher->dispatch($event);
            
            return $event->withdrawalId();
        }
        
        $id = $this->writer->insert($withdrawal);
        
        $event = StoredSingleWithdrawal::create(WithdrawalIdModel::create($id));
        $this->eventDispatcher->dispatch($event);
        
        return $event->withdrawalId();
    }
    
    
    /**
     * Stores all provided withdrawals.
     *
     * @param Withdrawals $withdrawals
     *
     * @return WithdrawalIds
     *
     * @throws ConnectionException
     */
    public function storeMultiple(Withdrawals $withdrawals): WithdrawalIds
    {
        $ids       = [];
        $operation = null;
        
        /**
         * @var string     $reference
         * @var Withdrawal $withdrawal
         */
        foreach ($withdrawals as $reference => $withdrawal) {
            if ($withdrawal->id() !== null) {
                $ids[$reference] = $withdrawal->id();
                $operation       = 'update';
            } elseif ($operation === 'update') {
                throw new InvalidArgumentException('All provided withdrawals need to be new or existing. '
                                                   . 'Can not mix store operation for new and existing withdrawals.');
            } else {
                $operation = 'insert';
            }
        }
        
        $event = StorageOfMultipleWithdrawalsRequested::create($withdrawals);
        $this->eventDispatcher->dispatch($event);
        
        if ($operation === 'update') {
            $this->writer->updateMultiple($withdrawals);
            
            $event = StoredMultipleWithdrawals::create($this->mapper->mapWithdrawalIds($ids));
            $this->eventDispatcher->dispatch($event);
            
            return $event->withdrawalIds();
        }
        
        $ids = $this->writer->insertMultiple($withdrawals);
        
        $event = StoredMultipleWithdrawals::create($this->mapper->mapWithdrawalIds($ids));
        $this->eventDispatcher->dispatch($event);
        
        return $event->withdrawalIds();
    }
    
    
    /**
     * Deletes the withdrawal with the provided ID.
     *
     * @param WithdrawalId $id
     */
    public function delete(WithdrawalId $id): void
    {
        $event = DeletionOfSingleWithdrawalRequested::create($id);
        $this->eventDispatcher->dispatch($event);
        
        $this->writer->delete($id);
        
        $event = DeletedSingleWithdrawal::create($id);
        $this->eventDispatcher->dispatch($event);
    }
    
    
    /**
     * Deletes multiple withdrawals based on the provided IDs.
     *
     * @param WithdrawalIds $ids
     *
     * @throws ConnectionException
     */
    public function deleteMultiple(WithdrawalIds $ids): void
    {
        $event = DeletionOfMultipleWithdrawalsRequested::create($ids);
        $this->eventDispatcher->dispatch($event);
        
        $this->writer->deleteMultiple($ids);
        
        $event = DeletedMultipleWithdrawals::create($ids);
        $this->eventDispatcher->dispatch($event);
    }
}